<?php

namespace Modules\Platform\User\Datatables;

use Modules\Places\Entities\Place;
use Modules\Companies\Entities\Company;
use Yajra\DataTables\EloquentDataTable;
use Modules\Platform\User\Entities\User;
use Modules\Platform\Core\Helper\DataTableHelper;
use Modules\Platform\Core\Datatable\PlatformDataTable;

/**
 * Class UsersDataTable
 * @package Modules\Platform\User\Datatables
 */
class UsersDataTable extends PlatformDataTable
{
    const SHOW_URL_ROUTE = 'settings.users.show';


    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        $dataTable = new EloquentDataTable($query);

        if(Request()->route()->getPrefix() == 'users/'){
            $this->applyLinks($dataTable, 'users.show');
        }
        else{
            $this->applyLinks($dataTable, self::SHOW_URL_ROUTE);
        }

        return $dataTable;
    }

    /**
     * Get query source of dataTable.
     *
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(User $model)
    {
 
        $query = $model->newQuery()
        ->leftJoin('companies', 'users.company_id', '=', 'companies.id')
        ->leftJoin('places', 'users.place_id', '=', 'places.id')
        ->select([
            'users.id',
            'users.company_id',
            'users.place_id',
            'companies.name as company_name',
            'places.identifier_name as place_name',
            'users.name',
            'users.email',
            'users.person_tax_number',
            'users.home_phone',
            'users.mobile_phone',
            'users.address_city',
            'users.address_state',
        ]);

        if (\Auth::user()->hasRole('chefe')) {
            $query = $query->where('users.company_id',\Auth::user()->company_id);
        }
        elseif(\Auth::user()->hasRole('gerente')){
            return null;
        }

        return $query;
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {
        return $this->builder()
            ->columns($this->getColumns())

            ->setTableAttribute('class', 'table table-hover')
            ->parameters([
                'dom' => "<'row'<'col-sm-12'tr>>\n\t\t\t<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7 dataTables_pager'lp>>",
                'responsive' => true,
                'stateSave' => true,
                'columnFilters' => $this->getFilters()
            ]);
    }


    /**
     * @return array
     */
    protected function getFilters(){

        $currentColumn = 0;
        $columnFilters = [];

        if(\Auth::user()->hasRole('admin')){

            $data_companies = [];
            foreach (Company::all()->toArray() as $company){
                $data_companies[] = [
                    'value' => $company['id'],
                    'label' => $company['name'],
                ];
            }

            $columnFilters[] = [
                'column_number' => $currentColumn++,
                'filter_type' => 'select',
                'select_type' => 'select2',
                'select_type_options' => [
                    'theme' => "bootstrap",
                    'width' => '100%'
                ],
                'data' => $data_companies                    
            ];
        }
        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){

            $places = [];
            $data_places = [];

            if (\Auth::user()->hasRole('admin')) {
                $places = Place::all()->toArray();
            }
            else {
                $places = Place::where('company_id', \Auth::user()->company_id)->get()->toArray();
            }

            foreach ($places as $place) {
                $data_places[] = [
                    'value' => $place['id'],
                    'label' => $place['identifier_name'],
                ];
            }

            $columnFilters[] = [
                'column_number' => $currentColumn++,
                'filter_type' => 'select',
                'select_type' => 'select2',
                'select_type_options' => [
                    'theme' => "bootstrap",
                    'width' => '100%'
                ],
                'data' => $data_places                    
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];

        return $columnFilters;
    }
    
    /**
     * @return array
     */
    protected function getColumns() {
    
        $data = [];

        if(\Auth::user()->hasRole('admin')){
            $data += [
                'company_name' => [
                    'name' => 'companies.name',
                    'data' => 'company_name',
                    'title' => trans('user::users.table.company_name'),
                    'data_type' => 'text'
                ], 
            ];
        }
        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){
            $data += [
                'place_name' => [
                    'name' => 'places.identifier_name',
                    'data' => 'place_name',
                    'title' => trans('user::users.table.place_name'),
                    'data_type' => 'text'
                ],
            ];
        }

        $data += [

            'name' => [
                'data' => 'name', 
                'title' => trans('user::users.table.first_name'), 
                'data_type' => 'text'
            ],
            'email' => [
                'data' => 'email', 
                'title' => trans('user::users.table.email'), 
                'data_type' => 'email'
            ],
            'person_tax_number' => [
                'data' => 'person_tax_number', 
                'title' => trans('user::users.table.person_tax_number'), 
                'data_type' => 'text'
            ],
            'home_phone' => [
                'data' => 'home_phone', 
                'title' => trans('user::users.table.home_phone'), 
                'data_type' => 'text'
            ],
            'mobile_phone' => [
                'data' => 'mobile_phone', 
                'title' => trans('user::users.table.mobile_phone'), 
                'data_type' => 'text'
            ],
            'address_city' => [
                'data' => 'address_city', 
                'title' => trans('user::users.table.address_city'), 
                'data_type' => 'text'
            ],
            'address_state' => [
                'data' => 'address_state', 
                'title' => trans('user::users.table.address_state'), 
                'data_type' => 'text'
            ],

        ];

        return $data;

    }
}
