<?php

namespace Modules\Platform\Core\Helper;

use Modules\Platform\User\Entities\Group;
use Modules\Platform\User\Entities\User;

/**
 * DataTable helper
 *
 * Class DataTableHelper
 * @package Modules\Platform\Core\Helper
 */
class DataTableHelper
{
    const FILTER_DROP_DOWN_CACHE = 1440;

    public static function buttons($title = 'Exported records')
    {
        return
            [
                [
                    'extend' => 'copy',
                    'title' => $title,
                    'exportOptions' => [
                        'format' => [
                            'header' => "function(mDataProp,columnIdx) { 
                                var htmlText = '<span>' + mDataProp + '</span>';
                                var jHtmlObject = jQuery(htmlText);
                                jHtmlObject.find('div').remove();
                                var newHtml = jHtmlObject.text();
                                return newHtml;
                                }"
                        ]
                    ]
                ],
                [
                    'extend' => 'print',
                    'title' => $title,
                    'exportOptions' => [
                        'format' => [
                            'header' => "function(mDataProp,columnIdx) { 
                                var htmlText = '<span>' + mDataProp + '</span>';
                                var jHtmlObject = jQuery(htmlText);
                                jHtmlObject.find('div').remove();
                                var newHtml = jHtmlObject.text();
                                return newHtml;
                                }"
                        ]
                    ]
                ],
                [
                    'extend' => 'excelHtml5',
                    'title' => $title,
                    'exportOptions' => [
                        'format' => [
                            'header' => "function(mDataProp,columnIdx) { 
                                var htmlText = '<span>' + mDataProp + '</span>';
                                var jHtmlObject = jQuery(htmlText);
                                jHtmlObject.find('div').remove();
                                var newHtml = jHtmlObject.text();
                                return newHtml;
                                }"
                        ]
                    ]
                ],
                [
                    'extend' => 'pdfHtml5',
                    'title' => $title,
                    'orientation' => 'landscape',
                    'exportOptions' => [
                        'format' => [
                            'header' => "function(mDataProp,columnIdx) { 
                                var htmlText = '<span>' + mDataProp + '</span>';
                                var jHtmlObject = jQuery(htmlText);
                                jHtmlObject.find('div').remove();
                                var newHtml = jHtmlObject.text();
                                return newHtml;
                                }"
                        ]
                    ]
                ]
            ];
    }

    /**
     *
     * @return array
     */
    public static function filterOwnerDropdown()
    {
        $result = [];

        $users = \Cache::remember('filter_dropdown_users', self::FILTER_DROP_DOWN_CACHE, function () {
            return User::all();
        });

        foreach ($users as $user) {
            $result[] = [
                'value' => $user->name,
                'label' => $user->name
            ];
        }

        $groups = \Cache::remember('filter_dropdown_groups', self::FILTER_DROP_DOWN_CACHE, function () {
            return Group::all();
        });

        foreach ($groups as $group) {
            $result[] = [
                'value' => $group->name,
                'label' => $group->name
            ];
        }

        return $result;
    }

    public static function queryCauser($query, $keyword)
    {
        $users = User::where('name', 'LIKE', '%' . $keyword . '%')->get()->pluck('id')->toArray();

        if (count($users) > 0) {
            $query->whereIn('causer_id', $users);
        }
    }

    /**
     * @param $query
     * @param $keyword
     * @param null $tableName - Use this where joining with others table
     */
    public static function queryOwner($query, $keyword, $tableName = null)
    {
        $users = User::where('name', 'LIKE', '%' . $keyword . '%')->get()->pluck('id')->toArray();
        $groups = Group::where('name', 'LIKE', '%' . $keyword . '%')->get()->pluck('id')->toArray();

        if ($tableName != null) {
            $tableName = $tableName . '.';
        }

        if (count($users) > 0 && count($groups) > 0) {
            $query->where(function ($q) use ($tableName) {
                $q->where($tableName . 'owned_by_type', Group::class);
                $q->orWhere($tableName . 'owned_by_type', User::class);
            });
        } else {
            if (count($users) > 0) {
                $query->where(function ($q) use ($tableName) {
                    $q->orWhere($tableName . 'owned_by_type', User::class);
                });
            } else {
                if (count($groups) > 0) {
                    $query->where(function ($q) use ($tableName) {
                        $q->orWhere($tableName . 'owned_by_type', Group::class);
                    });
                }
            }
        }

        $array = array_merge($users, $groups);
        $query->whereIn($tableName . 'owned_by_id', $array);
    }

    /**
     * @param $keyword
     * @return array|mixed
     */
    public static function getDatesForFilter($keyword)
    {
        if (strpos($keyword, ' - ') !== false) {
            $dates = str_replace('%%', '', $keyword);

            $dates = explode(" - ", $dates);

            if ($dates[0] != '') {
                $dates[0] = DateHelper::formatDateToUTC($dates[0] . ' 00:00:00');
            }
            if ($dates[1] != '') {
                $dates[1] = DateHelper::formatDateToUTC($dates[1] . ' 23:59:59');
            }

            return $dates;
        }
    }

    /**
     * Render column link
     *
     * @param $column
     * @param $record
     * @param $columnProperties
     * @param $route
     * @return string
     */
    public static function renderLink($column, $record, $columnProperties, $route, $eRoute = null, $dRoute = null)
    {
        $displayColumn = $record->$column;

        if ($route != '') {
            $href = route($route, $record->id);
        } else {
            $href = '#';
        }

        $datatype = 'text';

        if (isset($columnProperties['data_type'])) {
            $datatype = $columnProperties['data_type'];
        }

        switch ($datatype) {
            case 'boolean':
                $displayColumn = trans('core::core.yes_no.' . $displayColumn);
                break;

            case 'none':
                return $displayColumn;
                break;

            case 'datetime':
                $displayColumn = UserHelper::formatUserDateTime($displayColumn);
                break;

            case 'assigned_to':
                if ($record->owned_by != null) {
                    $displayColumn = $record->owned_by->name;
                }
                break;

            case 'email':
                $href = 'mailto:' . $record->$column;
                break;

            case 'date':
                $displayColumn = UserHelper::formatUserDate($displayColumn);
                break;

            case 'actions':
                return self::createActionsButton($record->id, $eRoute, $dRoute);
                break;

            default:
                # code...
                break;
        }

        /**
         * Add span on status columns
         */
        if($column == 'status_name'){

            switch (strip_tags($displayColumn)) {

                case 'Ativo':
                    $spanColor = "m-badge--success";
                    break;
    
                case 'Em manutenção':
                    $spanColor = "m-badge--warning";
                    break;
    
                case 'Desabilitado':
                    $spanColor = "m-badge--danger";
                    break;
        
                default:
                    $spanColor = "";
                    break;
            }
            
            $link = '<a data-column="' . strip_tags($column) . '" title="' . strip_tags($displayColumn) . '" href="' . $href . '"><span class="m-badge  '.$spanColor.' m-badge--wide"> ' . strip_tags($displayColumn) . '</span></a>';
        }

        /**
         * Add span on conservation_state columns
         */
        elseif($column == 'conservation_state'){

            switch (strip_tags($displayColumn)) {
                case 'perfeito':
                    $spanColor = "m-badge--perfeito";
                    break;

                case 'bom':
                    $spanColor = "m-badge--bom";
                    break;

                case 'regular':
                    $spanColor = "m-badge--regular";
                    break;

                case 'ruim':
                    $spanColor = "m-badge--ruim";
                    break;

                case 'muito ruim':
                    $spanColor = "m-badge--muito-ruim";
                    break;

                default:
                    $spanColor = "";
                    break;
            }

            $link = '<a data-column="' . strip_tags($column) . '" title="' . strip_tags($displayColumn) . '" href="' . $href . '"><span class="m-badge  '.$spanColor.' m-badge--wide"> ' . strip_tags($displayColumn) . '</span></a>';
        }


        else{
            $link = '<a data-column="' . strip_tags($column) . '" title="' . strip_tags($displayColumn) . '" href="' . $href . '"> ' . strip_tags($displayColumn) . '</a>';
        }

        return $link;
    }

    public static function createActionsButton($record, $editRoute, $deleteRoute)
    {

        $deleteFormView = \View::make('partial.delete-entity-action-form');

        $deleteFormView->with('dRoute', $deleteRoute);
        $deleteFormView->with('entity', $record);

        $deleteForm = $deleteFormView->render();

        $html = '<span class="dropdown">
                    <a href="#" class="btn m-btn m-btn--hover-brand m-btn--icon m-btn--icon-only m-btn--pill" data-toggle="dropdown" aria-expanded="false">
                      <i class="la la-ellipsis-h"></i>
                    </a>
                    <div class="dropdown-menu dropdown-menu-right" x-placement="bottom-end" style="position: absolute; will-change: transform; top: 0px; left: 0px; transform: translate3d(-196px, 27px, 0px);">
                        <a class="dropdown-item" href="'. route($editRoute, $record) .'"><i class="la la-edit"></i>'. trans('core::core.crud.edit') .'</a>
                        <a class="dropdown-item remove-element" href="#" onclick="document.getElementById(\'delete-form-submit-button-'. $record .'\').click()"><i class="la la-remove"></i>'. trans('core::core.crud.delete') .'</a>
                    </div>
                </span>'
                . $deleteForm;

        return $html;
    }
}
