<?php

namespace Modules\Places\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;

use Modules\Places\Entities\Place;
use Modules\Asset\Service\AssetService;
use Modules\Places\Http\Forms\PlaceForm;
use Modules\Places\Datatables\PlaceDatatable;
use Modules\Supplier\Service\SupplierService;
use Modules\Platform\User\Service\UserService;
use Modules\Places\Http\Requests\PlacesRequest;
use Modules\Collaborator\Service\CollaboratorService;
use Modules\Places\Datatables\Tabs\PlaceUsersDatatable;
use Modules\Places\Datatables\Tabs\PlaceAssetsDatatable;
use Modules\Places\Datatables\Tabs\PlaceCollaboratorsDatatable;
use Modules\Platform\Core\Http\Controllers\ModuleCrudController;


class PlacesController extends ModuleCrudController {

    protected $datatable = PlaceDatatable::class;
    protected $formClass = PlaceForm::class;
    protected $storeRequest = PlacesRequest::class;
    protected $updateRequest = PlacesRequest::class;
    protected $entityClass = Place::class;

    protected $moduleName = 'places';

        /**
     * overwriten show view for add dashboard tab from Default Crud view
     * @var array
     */
    protected $views = [
        'index' => 'core::crud.module.index',
        'show' => 'places::show',
        'create' => 'core::crud.module.create',
        'edit' => 'core::crud.module.edit',
    ];


    protected $permissions = [
        'browse' => 'places.browse',
        'create' => 'places.create',
        'update' => 'places.update',
        'destroy' => 'places.destroy'
    ];

    protected $moduleSettingsLinks = [];

    protected $settingsPermission = 'places.settings';

    protected $showFields = [

        'information' => [

            'company_id' => [
                'type' => 'hidden',
                'col-class' => 'col-12'
            ],

            'identifier_name' => [
                'type' => 'text',
                'col-class' => 'col-12'
            ],
            
        ],
        'adress' => [
            
            'city' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'state' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'street' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'number' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'neighborhood' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'complement' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'postal_code' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
        ],

        'notes' => [

            'notes' => [
                'type' => 'text',
                'col-class' => 'col-12'
            ],
        ],

    ];

    protected $languageFile = 'places::places';

    protected $routes = [
        'index' => 'places.places.index',
        'create' => 'places.places.create',
        'show' => 'places.places.show',
        'edit' => 'places.places.edit',
        'store' => 'places.places.store',
        'destroy' => 'places.places.destroy',
        'update' => 'places.places.update'
    ];

    protected $relationTabs = [
        'collaborators'         => [
            'icon'        => 'compare_arrows',
            'permissions' => [
                'browse' => 'collaborators.browse',
                'update' => 'collaborators.update',
                'create' => 'collaborators.create',
            ],
            'datatable'   => [
                'datatable' => PlaceCollaboratorsDatatable::class,
            ],
            'route'       => [
                'linked'        => 'places.collaborators.linked',
                'create'        => 'collaborators.collaborators.create',
                'select'        => 'places.collaborators.selection',
                'bind_selected' => 'places.collaborators.link',
            ],
            'create'      => [
                'allow'            => true,
                'modal_title'      => 'collaborators::collaborators.create_new',
                'post_create_bind' => [
                    'relationType' => 'oneToMany',
                    'relatedField' => 'place_id',

                ],
            ],
            'select'      => [
                'allow'       => false,
                'modal_title' => 'collaborators::collaborators.module',
            ],
        ],
        'assets'         => [
            'icon'        => 'compare_arrows',
            'permissions' => [
                'browse' => 'assets.browse',
                'update' => 'assets.update',
                'create' => 'assets.create',
            ],
            'datatable'   => [
                'datatable' => PlaceAssetsDatatable::class,
            ],
            'route'       => [
                'linked'        => 'places.assets.linked',
                'create'        => 'assets.assets.create',
                'select'        => 'places.assets.selection',
                'bind_selected' => 'places.assets.link',
            ],
            'create'      => [
                'allow'            => true,
                'modal_title'      => 'assets::assets.create_new',
                'post_create_bind' => [
                    'relationType' => 'oneToMany',
                    'relatedField' => 'place_id',

                ],
            ],

            'select'      => [
                'allow'       => false,
                'modal_title' => 'assets::assets.module',
            ],
        ],
        'users'         => [
            'icon'        => 'compare_arrows',
            'permissions' => [
                'browse' => 'assets.browse',
                'update' => 'assets.update',
                'create' => 'assets.create',
            ],
            'datatable'   => [
                'datatable' => PlaceUsersDatatable::class,
            ],
            'route'       => [
                'linked'        => 'places.users.linked',
                'create'        => 'settings.users.create',
                'select'        => 'places.users.selection',
                'bind_selected' => 'places.users.link',
            ],
            'create'      => [
                'allow'            => true,
                'modal_title'      => 'places::places.create_new_user',
                'post_create_bind' => [
                    'relationType' => 'oneToMany',
                    'relatedField' => 'place_id',
                ],
            ],
            'select'      => [
                'allow'       => false,
                'modal_title' => 'users::users.module',
            ],
        ],

    ];

    public function __construct()
    {
        parent::__construct();

    }

    /**
     * Overwriten function for add dashboard info
     *
     * @param $identifier
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show($identifier) {

        if ($this->permissions['browse'] != '' && !\Auth::user()->hasPermissionTo($this->permissions['browse'])) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $repository = $this->getRepository();

        $entity = $repository->find($identifier);

        $this->entity = $entity;

        if (empty($entity)) {
            flash(trans('core::core.entity.entity_not_found'))->error();
            return redirect(route($this->routes['index']));
        }

        if ($this->blockEntityOwnableAccess()) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $this->entityIdentifier = $entity->id;

        $showView = $this->views['show'];

        $this->entity = $entity;

        $view = view($showView);
        $view->with('entity', $entity);
        $view->with('show_fields', $this->showFields);
        $view->with('show_fileds_count', count($this->showFields));

        $view->with('next_record', $repository->next($entity));
        $view->with('prev_record', $repository->prev($entity));
        $view->with('disableNextPrev', $this->disableNextPrev);

        $this->setupCustomButtons();
        $this->setupActionButtons();
        $view->with('customShowButtons', $this->customShowButtons);
        $view->with('actionButtons', $this->actionButtons);
        $view->with('commentableExtension', false);
        $view->with('actityLogDatatable', null);
        $view->with('attachmentsExtension', false);
        $view->with('entityIdentifier', $this->entityIdentifier);
        $view->with('moduleName', $this->moduleName);


        $view->with('hasExtensions', false);

        $view->with('relationTabs', $this->setupRelationTabs($entity));

        $view->with('baseIcons', $this->baseIcons);
        /*
         * Extensions
         */
        if (in_array(self::COMMENTS_EXTENSION, class_uses($this->entity))) {
            $view->with('commentableExtension', true);
            $view->with('hasExtensions', true);
        }
        if (in_array(self::ACTIVITY_LOG_EXTENSION, class_uses($this->entity))) {
            $activityLogDataTable = \App::make(ActivityLogDataTable::class);
            $activityLogDataTable->setEntityData(get_class($entity), $entity->id);
            $view->with('actityLogDatatable', $activityLogDataTable->html());
            $view->with('hasExtensions', true);
        }
        if (in_array(self::ATTACHMENT_EXTENSION, class_uses($this->entity))) {
            $view->with('attachmentsExtension', true);
            $view->with('hasExtensions', true);
        }

        /**
         * dashboard info
        **/
        $userService = \App::make(UserService::class);
        $collaboratorService = \App::make(CollaboratorService::class);
        $assetService = \App::make(AssetService::class);
        $supplierService = \App::make(SupplierService::class);

        $usersCount = $userService->countByPlace($entity->id);
        $collaboratorsCount = $collaboratorService->countByPlace($entity->id);
        $suppliersCount = $supplierService->countByCompany($entity->company_id);
        $assetsCount = $assetService->countByPlace($entity->id);

        $dataPieChart = $assetService->getByCategory(null,$entity->id);
        $dataContainerChart = $assetService->getBystatus(Place::find($entity->id)['company_id'],$entity->id);

        $lastUpdates = $assetService->lastUpdates("place", $entity->id);

        \JavaScript::put([
            'dataPieChart' => json_encode($dataPieChart),
            'dataContainerChart' => json_encode($dataContainerChart),
        ]);

        $view->with('usersCount', $usersCount);
        $view->with('collaboratorsCount', $collaboratorsCount);
        $view->with('assetsCount', $assetsCount);
        $view->with('suppliersCount', $suppliersCount);
        $view->with('dataPieChart', $dataPieChart);
        $view->with('dataContainerChart', $dataContainerChart);
        $view->with('lastUpdates', $lastUpdates);

        return $view;
    }


}
