<?php

namespace Modules\Companies\Http\Controllers;

use Modules\Places\Entities\Place;
use Modules\Assets\Entities\Status;
use Modules\Assets\Entities\Category;
use Modules\Asset\Service\AssetService;
use Modules\Companies\Entities\Company;
use Modules\Collaborators\Entities\Area;
use Modules\Suppliers\Entities\Supplier;
use Modules\Companies\Http\Forms\CompanyForm;
use Modules\Supplier\Service\SupplierService;
use Modules\Platform\Core\Helper\StringHelper;
use Modules\Platform\User\Service\UserService;
use Modules\Collaborators\Entities\Collaborator;
use Modules\Companies\Datatables\CompanyDatatable;
use Modules\Platform\Core\Helper\ValidationHelper;
use Modules\Collaborator\Service\CollaboratorService;
use Modules\Companies\Http\Requests\CompaniesRequest;
use Modules\Platform\Core\Repositories\GenericRepository;
use Modules\Companies\Datatables\Tabs\CompanyAssetsDatatable;
use Modules\Companies\Datatables\Tabs\CompanyPlacesDatatable;
use Modules\Platform\Core\Http\Controllers\ModuleCrudController;
use Modules\Companies\Datatables\Tabs\CompanyCollaboratorsDatatable;
use Modules\Platform\Notifications\Entities\NotificationPlaceholder;

class CompaniesController extends ModuleCrudController
{ 
 
    protected $datatable = CompanyDatatable::class;
    protected $formClass = CompanyForm::class;
    protected $storeRequest = CompaniesRequest::class;
    protected $updateRequest = CompaniesRequest::class;
    protected $entityClass = Company::class;

    protected $moduleName = 'companies';
 
    protected $permissions = [
        'browse' => 'companies.browse',
        'create' => 'companies.create',
        'update' => 'companies.update',
        'destroy' => 'companies.destroy'
    ];

    /**
     * overwriten show view for add dashboard tab from Default Crud view
     * @var array
     */
    protected $views = [
        'index' => 'core::crud.module.index',
        'show' => 'companies::show',
        'create' => 'core::crud.module.create',
        'edit' => 'core::crud.module.edit',
    ];

    protected $moduleSettingsLinks = [

    ];
 
    protected $relationTabs = [
        'places'         => [
            'icon'        => 'compare_arrows',
            'permissions' => [
                'browse' => 'places.browse',
                'update' => 'places.update',
                'create' => 'places.create',
            ],
            'datatable'   => [
                'datatable' => CompanyPlacesDatatable::class,
            ],
            'route'       => [
                'linked'        => 'companies.places.linked',
                'create'        => 'places.places.create',
                'select'        => 'companies.places.selection',
                'bind_selected' => 'companies.places.link',
            ],
            'create'      => [
                'allow'            => true,
                'modal_title'      => 'places::places.create_new',
                'post_create_bind' => [
                    'relationType' => 'oneToMany',
                    'relatedField' => 'company_id',
                ],
            ],
            'select'      => [
                'allow'       => false,
                'modal_title' => 'places::places.module',
            ],
        ],
        'collaborators'         => [
            'icon'        => 'compare_arrows',
            'permissions' => [
                'browse' => 'collaborators.browse',
                'update' => 'collaborators.update',
                'create' => 'collaborators.create',
            ],
            'datatable'   => [
                'datatable' => CompanyCollaboratorsDatatable::class,
            ],
            'route'       => [
                'linked'        => 'companies.collaborators.linked',
                'create'        => 'collaborators.collaborators.create',
                'select'        => 'companies.collaborators.selection',
                'bind_selected' => 'companies.collaborators.link',
            ],
            'create'      => [
                'allow'            => true,
                'modal_title'      => 'collaborators::collaborators.create_new',
                'post_create_bind' => [
                    'relationType' => 'oneToMany',
                    'relatedField' => 'company_id',

                ],
            ],

            'select'      => [
                'allow'       => false,
                'modal_title' => 'collaborators::collaborators.module',
            ],
        ],
        'assets'         => [
            'icon'        => 'compare_arrows',
            'permissions' => [
                'browse' => 'assets.browse',
                'update' => 'assets.update',
                'create' => 'assets.create',
            ],
            'datatable'   => [
                'datatable' => CompanyAssetsDatatable::class,
            ],
            'route'       => [
                'linked'        => 'companies.assets.linked',
                'create'        => 'assets.assets.create',
                'select'        => 'companies.assets.selection',
                'bind_selected' => 'companies.assets.link',
            ],
            'create'      => [
                'allow'            => true,
                'modal_title'      => 'assets::assets.create_new',
                'post_create_bind' => [
                    'relationType' => 'oneToMany',
                    'relatedField' => 'company_id',

                ],
            ],

            'select'      => [
                'allow'       => false,
                'modal_title' => 'assets::assets.module',
            ],
        ],
    ];

    protected $settingsPermission = 'companies.settings';

    protected $showFields = [

        'information' => [

            'name' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'company_tax_number' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'email' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'phone' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'mobile' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
        ],
        'address_information' => [
            'state' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'city' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'street' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'number' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'neighborhood' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'complement' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'postal_code' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

        ],

        'notes' => [

            'notes' => [
                'type' => 'text',
                'col-class' => 'col-12'
            ],

        ],

    ];

    protected $languageFile = 'companies::companies';

    protected $routes = [
        'index' => 'companies.companies.index',
        'create' => 'companies.companies.create',
        'show' => 'companies.companies.show',
        'edit' => 'companies.companies.edit',
        'store' => 'companies.companies.store',
        'destroy' => 'companies.companies.destroy',
        'update' => 'companies.companies.update'
    ];

    public function __construct()
    {
        parent::__construct();

    }

    /**
     * Overwriten function for add tab dashboard info
     *
     * @param $identifier
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show($identifier) {

        if ($this->permissions['browse'] != '' && !\Auth::user()->hasPermissionTo($this->permissions['browse'])) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $repository = $this->getRepository();

        $entity = $repository->find($identifier);

        $this->entity = $entity;

        if (empty($entity)) {
            flash(trans('core::core.entity.entity_not_found'))->error();
            return redirect(route($this->routes['index']));
        }

        if ($this->blockEntityOwnableAccess()) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $this->entityIdentifier = $entity->id;

        $showView = $this->views['show'];

        $this->entity = $entity;

        $view = view($showView);
        $view->with('entity', $entity);
        $view->with('show_fields', $this->showFields);
        $view->with('show_fileds_count', count($this->showFields));

        $view->with('next_record', $repository->next($entity));
        $view->with('prev_record', $repository->prev($entity));
        $view->with('disableNextPrev', $this->disableNextPrev);

        $this->setupCustomButtons();
        $this->setupActionButtons();
        $view->with('customShowButtons', $this->customShowButtons);
        $view->with('actionButtons', $this->actionButtons);
        $view->with('commentableExtension', false);
        $view->with('actityLogDatatable', null);
        $view->with('attachmentsExtension', false);
        $view->with('entityIdentifier', $this->entityIdentifier);
        $view->with('moduleName', $this->moduleName);


        $view->with('hasExtensions', false);

        $view->with('relationTabs', $this->setupRelationTabs($entity));

        $view->with('baseIcons', $this->baseIcons);
        /*
         * Extensions
         */
        if (in_array(self::COMMENTS_EXTENSION, class_uses($this->entity))) {
            $view->with('commentableExtension', true);
            $view->with('hasExtensions', true);
        }
        if (in_array(self::ACTIVITY_LOG_EXTENSION, class_uses($this->entity))) {
            $activityLogDataTable = \App::make(ActivityLogDataTable::class);
            $activityLogDataTable->setEntityData(get_class($entity), $entity->id);
            $view->with('actityLogDatatable', $activityLogDataTable->html());
            $view->with('hasExtensions', true);
        }
        if (in_array(self::ATTACHMENT_EXTENSION, class_uses($this->entity))) {
            $view->with('attachmentsExtension', true);
            $view->with('hasExtensions', true);
        }

        /**
         * dashboard info
        **/
        $userService = \App::make(UserService::class);
        $collaboratorService = \App::make(CollaboratorService::class);
        $assetService = \App::make(AssetService::class);
        $supplierService = \App::make(SupplierService::class);

        $usersCount = $userService->countByCompany($entity->id);
        $collaboratorsCount = $collaboratorService->countByCompany($entity->id);
        $suppliersCount = $supplierService->countByCompany($entity->id);
        $assetsCount = $assetService->countByCompany($entity->id);

        $dataPieChart = $assetService->getByCategory($entity->id);
        $dataContainerChart = $assetService->getBystatus($entity->id,null);

        $lastUpdates = $assetService->lastUpdates("company", $entity->id);

        \JavaScript::put([
            'dataPieChart' => json_encode($dataPieChart),
            'dataContainerChart' => json_encode($dataContainerChart),
        ]);

        $view->with('usersCount', $usersCount);
        $view->with('collaboratorsCount', $collaboratorsCount);
        $view->with('assetsCount', $assetsCount);
        $view->with('suppliersCount', $suppliersCount);
        $view->with('dataPieChart', $dataPieChart);
        $view->with('dataContainerChart', $dataContainerChart);
        $view->with('lastUpdates', $lastUpdates);

        return $view;
    }


    /**
     * Overwriten function for add company predefined status
     *
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store() {

        $request = \App::make($this->storeRequest ?? Request::class);

        $mode = $request->get('entityCreateMode', self::FORM_MODE_FULL);

        if ($this->permissions['create'] != '' && !\Auth::user()->hasPermissionTo($this->permissions['create'])) {
            if ($mode == self::FORM_MODE_SIMPLE) {
                return response()->json([
                    'type' => 'error',
                    'message' => trans('core::core.entity.you_dont_have_access'),
                    'action' => 'show_message'
                ]);
            }
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $repository = $this->getRepository();

        $storeValues = $this->form($this->formClass)->getFieldValues(true);

        if ($mode == self::FORM_MODE_SIMPLE) {

            //Bind related element
            $relatedEntityId = $request->get('relatedEntityId');
            $relationType = $request->get('relationType', null);
            $relatedField = $request->get('relatedField');
            $relatedEntity = $request->get('relatedEntity');

            if ($relationType != null) { // Relation type is not null

                $relationEntityRepos = \App::make(GenericRepository::class);
                $relationEntityRepos->setupModel($relatedEntity);

                $relationEntity = $relationEntityRepos->findWithoutFail($relatedEntityId);

                if ($relationType == 'oneToMany') {
                    $storeValues[$relatedField] = $relationEntity->id;
                }
            }
        }

        $input = [];
        foreach($storeValues as $key => $value){
            if($value === ''){
                unset($storeValues[$key]);
            }
        }


        $entity = $repository->createEntity($storeValues, \App::make($this->entityClass));


        $entity = $this->setupAssignedTo($entity, $request, true);
        $entity->save();

        /**
         * Apply predefined status
         */
        $entity->status()->attach([1,2,3]);

        if (config('bap.record_assigned_notification_enabled')) {

            if ($entity instanceof Ownable) {
                if ($entity->getOwner() != null && $entity->getOwner() instanceof User) {
                    if ($entity->getOwner()->id != \Auth::user()->id) { // Dont send notification for myself
                        try {
                            $commentOn = $entity->name;
                            $commentOn = ' - ' . $commentOn;
                        } catch (\Exception $exception) {
                            $commentOn = '';
                        }

                        $placeholder = new NotificationPlaceholder();

                        $placeholder->setRecipient($entity->getOwner());
                        $placeholder->setAuthorUser(\Auth::user());
                        $placeholder->setAuthor(\Auth::user()->name);
                        $placeholder->setColor('bg-green');
                        $placeholder->setIcon('assignment');
                        $placeholder->setContent(trans('notifications::notifications.new_record', ['user' => \Auth::user()->name]) . $commentOn);

                        $placeholder->setUrl(route($this->routes['show'], $entity->id));

                        $entity->getOwner()->notify(new GenericNotification($placeholder));
                    }
                }
            }
        }

        if ($mode == self::FORM_MODE_SIMPLE) {

            //Bind related element
            $relatedEntityId = $request->get('relatedEntityId');
            $relationType = $request->get('relationType', null);
            $relatedField = $request->get('relatedField');
            $relatedEntity = $request->get('relatedEntity');

            if ($relationType != null) { // Relation type is not null

                $relationEntityRepos = \App::make(GenericRepository::class);
                $relationEntityRepos->setupModel($relatedEntity);


                $relationEntity = $relationEntityRepos->findWithoutFail($relatedEntityId);


                if ($relationType == 'manyToMany') {
                    $entity->{$relatedField}()->attach($relationEntity->id);
                }
            }

            return response()->json([
                'type' => 'success',
                'message' => trans('core::core.entity.created'),
                'action' => 'refresh_datatable'
            ]);
        }

        if(! isset($this->settingName) ){
            $msg = $this->moduleName. '::' .$this->moduleName.'.created';
        }
        else{
            $msg = $this->moduleName. '::' .$this->moduleName.'.'.$this->settingName.'.created';
        }

        flash(trans($msg))->success();

        return redirect(route($this->routes['index']));
    }


    /**
     * Overwriten function for remove company attached status
     *
     * @param $identifier
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function destroy($identifier) {

        if ($this->permissions['destroy'] != '' && !\Auth::user()->hasPermissionTo($this->permissions['destroy'])) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $repository = $this->getRepository();

        $entity = $repository->find($identifier);

        $this->entity = $entity;

        if (empty($entity)) {
            flash(trans('core::core.entity.entity_not_found'))->error();
            return redirect(route($this->routes['index']));
        }

        if ($this->blockEntityOwnableAccess()) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        if (config('bap.validate_fk_on_soft_delete')) {
            $validator = ValidationHelper::validateForeignKeys($entity);

            if (count($validator) > 0) {
                flash(trans('core::core.cant_delete_check_fk_keys', ['fk_keys' => StringHelper::validationArrayToString($validator)]))->error();
                return redirect(route($this->routes['index']));
            }
        }

        try {
            /**
             * Removing status
             */
            $entity->status()->detach();

            $repository->delete($entity->id);

        } catch(\Exception $exception){

            flash("Impossível deletar, possui dependências")->error();

            return back();
        }

        if(! isset($this->settingName) ){
            $msg = $this->moduleName. '::' .$this->moduleName.'.deleted';
        }
        else{
            $msg = $this->moduleName. '::' .$this->moduleName.'.'.$this->settingName.'.deleted';
        }
        
        flash(trans($msg))->success();

        return redirect(route($this->routes['index']));
    }


    public function getPlaces($company_id){

        $places = Place::where('company_id', $company_id)->pluck('identifier_name', 'id')->toArray();

        return response()->json($places);
    }

    public function getSuppliers($company_id){

        $suppliers = Supplier::where('company_id', $company_id)->pluck('fantasy_name', 'id')->toArray();

        return response()->json($suppliers);
    }

    public function getAreas($company_id){

        $areas = Area::where('company_id', $company_id)->pluck('name', 'id')->toArray();

        return response()->json($areas);
    }

    public function getStatus($company_id){

        $status = Company::find($company_id)->status()->pluck('status.name', 'status.id')->toArray();

        return response()->json($status);
    }

    public function getCollaborators($company_id){

        $collaborators = Collaborator::where('company_id',$company_id)->pluck('name', 'id')->toArray();

        return response()->json($collaborators);
    }

    public function getCategories($company_id){

        $categories = Category::where('company_id',$company_id)->pluck('name', 'id')->toArray();

        return response()->json($categories);
    }


}
