<?php

namespace Modules\Clients\Datatables;


use Modules\Places\Entities\Place;
use Modules\Clients\Entities\Client;
use Modules\Companies\Entities\Company;
use Yajra\DataTables\EloquentDataTable;
use Modules\Platform\Core\Helper\DataTableHelper;
use Modules\Platform\Core\Datatable\PlatformDataTable;

class ClientDatatable extends PlatformDataTable
{

    const SHOW_URL_ROUTE = 'clients.clients.show';
    const EDIT_URL_ROUTE = 'clients.clients.edit';
    const REMOVE_URL_ROUTE = 'clients.clients.destroy';

    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        $dataTable = new EloquentDataTable($query);

        $this->applyLinks($dataTable, self::SHOW_URL_ROUTE, self::EDIT_URL_ROUTE, self::REMOVE_URL_ROUTE);

        return $dataTable;
    }

    /**
     * Get query source of dataTable.
     *
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(Client $model)
    {

        $query = $model->newQuery()
        ->leftJoin('companies', 'clients.company_id', '=', 'companies.id')
        ->leftJoin('places', 'clients.place_id', '=', 'places.id')
        ->select([
            'clients.id',
            'clients.company_id',
            'clients.place_id',
            'companies.name as company_name',
            'places.identifier_name as place_name',
            'clients.fantasy_name',
            'clients.person_type',
            'clients.person_tax_number-company_tax_number',
            'clients.email',
            'clients.phone',
            'clients.phone',
            'clients.mobile',
            'clients.city',
            'clients.state',
        ]);

        if (\Auth::user()->hasRole('chefe')) {
            $query = $query->where('clients.company_id',\Auth::user()->company_id);
        }
        elseif(\Auth::user()->hasRole('gerente')){
            $query = $query->where('clients.place_id',\Auth::user()->place_id);
        }

        return $query;
    }
    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {        
        return $this->builder()
            ->columns($this->getColumns())

            ->setTableAttribute('class', 'table responsive table-hover')
            ->parameters([
                'dom' => "<'row'<'col-sm-12'tr>>\n\t\t\t<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7 dataTables_pager'lp>>",

                'stateSave' => true,
                'columnFilters' => $this->getFilters(),
                'buttons' => DataTableHelper::buttons(),
                'regexp' => true

            ]);
    }

    /**
     * @return array
     */
    protected function getFilters(){

        $currentColumn = 0;
        $columnFilters = [];

        if(\Auth::user()->hasRole('admin')){

            $data_companies = [];
            foreach (Company::all()->toArray() as $company){
                $data_companies[] = [
                    'value' => $company['id'],
                    'label' => $company['name'],
                ];
            }

            $columnFilters[] = [
                    'column_number' => $currentColumn++,
                    'filter_type' => 'select',
                    'select_type' => 'select2',
                    'select_type_options' => [
                        'theme' => "bootstrap",
                        'width' => '100%'
                    ],
                    'data' => $data_companies                    
                ];
        }
        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){

            $places = [];
            $data_places = [];
            if (\Auth::user()->hasRole('admin')) {
                $places = Place::all()->toArray();
            }
            else {
                $places = Place::where('company_id', \Auth::user()->company_id)->get()->toArray();
            }

            foreach ($places as $place) {
                $data_places[] = [
                    'value' => $place['id'],
                    'label' => $place['identifier_name'],
                ];
            }

            $columnFilters[] = [
                    'column_number' => $currentColumn++,
                    'filter_type' => 'select',
                    'select_type' => 'select2',
                    'select_type_options' => [
                        'theme' => "bootstrap",
                        'width' => '100%'
                    ],
                    'data' => $data_places                    
                ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];

        return $columnFilters;
    }


    /**
     * @return array
     */
    protected function getColumns()
    {

        $data = [];

        if(\Auth::user()->hasRole('admin')){
            $data += [
                'company_name' => [
                    'name' => 'company_id',
                    'data' => 'company_name',
                    'title' => trans('clients::clients.table.company_name'),
                    'data_type' => 'text'
                ],
            ];
        }
        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){
            $data += [
                'place_name' => [
                    'name' => 'place_id',
                    'data' => 'place_name',
                    'title' => trans('clients::clients.table.place_name'),
                    'data_type' => 'text'
                ],
            ];
        }

        $data += [

            'fantasy_name' => [
                'data' => 'fantasy_name',
                'title' => trans('clients::clients.table.fantasy_name'),
                'data_type' => 'text'
            ],
            'person_type' => [
                'data' => 'person_type',
                'title' => trans('clients::clients.table.person_type'),
                'data_type' => 'text'
            ],
            'person_tax_number-company_tax_number' => [
                'data' => 'person_tax_number-company_tax_number',
                'title' => trans('clients::clients.table.person_tax_number-company_tax_number'),
                'data_type' => 'text'
            ],
            'email' => [
                'data' => 'email',
                'title' => trans('clients::clients.table.email'),
                'data_type' => 'email'
            ],
            'phone' => [
                'data' => 'phone',
                'title' => trans('clients::clients.table.phone'),
                'data_type' => 'text'
            ],
            'mobile' => [
                'data' => 'mobile',
                'title' => trans('clients::clients.table.mobile'),
                'data_type' => 'text'
            ],
            'city' => [
                'data' => 'city',
                'title' => trans('clients::clients.table.city'),
                'data_type' => 'text'
            ],
            'state' => [
                'data' => 'state',
                'title' => trans('clients::clients.table.state'),
                'data_type' => 'text'
            ],
            'actions' => ['title' => trans('core::core.table.actions'), 'data_type' => 'actions', 'orderable' => false],
        ];

        return $data;

    }

}
