<?php

namespace Modules\Assets\Datatables;


use Modules\Assets\Entities\Asset;
use Modules\Places\Entities\Place;
use Modules\Assets\Entities\Status;
use Modules\Assets\Entities\Category;
use Modules\Companies\Entities\Company;
use Yajra\DataTables\EloquentDataTable;
use Modules\Collaborators\Entities\Area;
use Modules\Collaborators\Entities\Collaborator;
use Modules\Platform\Core\Helper\DataTableHelper;
use Modules\Platform\Core\Datatable\PlatformDataTable;
 
class AssetDatatable extends PlatformDataTable
{

    const SHOW_URL_ROUTE = 'assets.assets.show';
    const EDIT_URL_ROUTE = 'assets.assets.edit';
    const REMOVE_URL_ROUTE = 'assets.assets.destroy';

    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        $dataTable = new EloquentDataTable($query);

        $this->applyLinks($dataTable, self::SHOW_URL_ROUTE, self::EDIT_URL_ROUTE, self::REMOVE_URL_ROUTE);

        return $dataTable;
    }

    /**
     * Get query source of dataTable.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */ 
    public function query(Asset $model) {

        $query = $model->newQuery()
        ->leftJoin('categories', 'assets.category_id', '=', 'categories.id')
        ->leftJoin('status', 'assets.status_id', '=', 'status.id')
        ->leftJoin('areas', 'assets.area_id', '=', 'areas.id')
        ->leftJoin('places', 'assets.place_id', '=', 'places.id')
        ->leftJoin('companies', 'assets.company_id', '=', 'companies.id')
        ->leftJoin('collaborators', 'assets.responsible', '=', 'collaborators.id')
        ->select([
            'assets.id',
            'assets.company_id',
            'assets.status_id',
            'assets.area_id',
            'assets.area_id',
            'assets.category_id',
            'assets.place_id',
            'assets.conservation_state',
            'assets.asset_number',
            'assets.serial_number_1',
            'assets.item',
            'categories.name as category_name',
            'areas.name as area_name',
            'assets.responsible',
            'places.identifier_name as place_name',
            'status.name as status_name',
            'companies.name as company_name',
            'collaborators.name as collaborator_name'
        ]);

        if (\Auth::user()->hasRole('chefe')) {
            $query = $query->where('assets.company_id',\Auth::user()->company_id);
        }
        else if(\Auth::user()->hasRole('gerente')){
            $query = $query->where('assets.place_id',\Auth::user()->place_id);
        }

        return $query;
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html() {

        return $this->builder()
            ->columns($this->getColumns())
            ->setTableAttribute('class', 'table table-hover')
            ->parameters([
                'dom' => "<'row'<'col-sm-12'tr>>\n\t\t\t<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7 dataTables_pager'lp>>",
                'stateSave' => true,
                'columnFilters' => $this->getFilters(),
                'buttons' => DataTableHelper::buttons(),
                'regexp' => true
            ]);
    }

    /**
     * @return array
     */
    protected function getFilters(){

        $currentColumn = 0;
        $columnFilters = [];

        if(\Auth::user()->hasRole('admin')){

            $data_companies = [];
            foreach (Company::all()->toArray() as $company){
                $data_companies[] = [
                    'value' => $company['id'],
                    'label' => $company['name'],
                ];
            }

            $columnFilters[] = [
                    'column_number' => $currentColumn++,
                    'filter_type' => 'select',
                    'select_type' => 'select2',
                    'select_type_options' => [
                        'theme' => "bootstrap",
                        'width' => '100%'
                    ],
                    'data' => $data_companies                    
                ];
        }
        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){

            $places = [];
            $data_places = [];

            if (\Auth::user()->hasRole('admin')) {
                $places = Place::all()->toArray();
            }
            else {
                $places = Place::where('company_id', \Auth::user()->company_id)->get()->toArray();
            }

            foreach ($places as $place) {
                $data_places[] = [
                    'value' => $place['id'],
                    'label' => $place['identifier_name'],
                ];
            }

            $columnFilters[] = [
                'column_number' => $currentColumn++,
                'filter_type' => 'select',
                'select_type' => 'select2',
                'select_type_options' => [
                    'theme' => "bootstrap",
                    'width' => '100%'
                ],
                'data' => $data_places                    
            ];
        }

        $areas = [];

        $data_areas = [];

        $areas = Area::all()->toArray();

        foreach ($areas as $area){
            $data_areas[] = [
                'value' => $area['id'],
                'label' => $area['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_areas                    
        ];

        $categories = [];
        $data_categories = [];

        $categories = Category::all()->toArray();

        foreach ($categories as $category){
            $data_categories[] = [
                'value' => $category['id'],
                'label' => $category['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_categories                    
        ];

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        
        $status = [];
        $data_status = [];

        $status = Status::all()->toArray();

        foreach ($status as $stat){
            $data_status[] = [
                'value' => $stat['id'],
                'label' => $stat['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_status                    
        ];

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];

        $collaborators = [];
        $data_collaborators = [];
        if (\Auth::user()->hasRole('admin')) {
            $collaborators = Collaborator::all()->toArray();
        }
        else {
            $collaborators = Collaborator::where('company_id', \Auth::user()->company_id)->get()->toArray();
        }

        foreach ($collaborators as $collaborator){
            $data_collaborators[] = [
                'value' => $collaborator['id'],
                'label' => $collaborator['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_collaborators
        ];

        return $columnFilters;
    }

    /**
     * @return array
     */
    protected function getColumns()
    {
        $data = [];

        if(\Auth::user()->hasRole('admin')){
            $data += [
                'company_name' => [
                    'name' => 'company_id',
                    'data' => 'company_name',
                    'title' => trans('assets::assets.table.company_name'),
                    'data_type' => 'text'
                ],
            ];
        }
        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){
            $data += [
                'place_name' => [
                    'name' => 'place_id',
                    'data' => 'place_name',
                    'title' => trans('assets::assets.table.place_name'),
                    'data_type' => 'text'
                ],
            ];
        }

        $data += [

            'area_name' => [
                'name' => 'area_id',
                'data' => 'area_name',
                'title' => trans('assets::assets.table.area_name'),
                'data_type' => 'text'
            ],
            'category_name' => [
                'name' => 'category_id',
                'data' => 'category_name',
                'title' => trans('assets::assets.table.category_name'),
                'data_type' => 'text'
            ],
            'item' => [
                'data' => 'item',
                'title' => trans('assets::assets.table.item'),
                'data_type' => 'text',
            ],
            'status_name' => [
                'name' => 'status_id',
                'data' => 'status_name',
                'title' => trans('assets::assets.table.status_id'),
                'data_type' => 'text'
            ],
            'conservation_state' => [
                'data' => 'conservation_state',
                'title' => trans('assets::assets.table.conservation_state'),
                'data_type' => 'text',
            ],
            'asset_number' => [
                'data' => 'asset_number',
                'title' => trans('assets::assets.table.asset_number'),
                'data_type' => 'text'
            ],
            'serial_number_1' => [
                'data' => 'serial_number_1',
                'title' => trans('assets::assets.table.serial_number_1'),
                'data_type' => 'text'
            ],
            'responsible' => [
                'name' => 'responsible',
                'data' => 'collaborator_name',
                'title' => trans('assets::assets.table.responsible'),
                'data_type' => 'text'
            ],

            'actions' => ['title' => trans('core::core.table.actions'), 'data_type' => 'actions', 'orderable' => false],
        ];

        return $data;

    }

}
